#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/check_rp_filter.sh /main/2 2011/05/04 16:58:00 ptare Exp $
#
# check_rp_filter.sh
#
# Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      check_rp_filter.sh - check the reverse path filter setting for cluster private interconnect classified NICs 
#
#    DESCRIPTION
#      check the reverse path filter "rp_filter" parameter for NICS selected for private interconnect 
#
#    NOTES
#      Currently this check only applies to OEL6 and further LINUX releases 
#
#    MODIFIED   (MM/DD/YY)
#    ptare       04/27/11 - check the reverse path filter rp_filter for NICS on
#                           LINUX
#    ptare       04/27/11 - Creation
#
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin
SECHO="echo"
SGREP="grep"
SAWK="awk"
SSYSCTL="/sbin/sysctl -q"
SEPARATOR=","
_HOST=`/bin/hostname`

# Set default exit status to indicate failure.
existstatus=3
badNicList=""
expected="0|2"

#method to get the value of rp_flter kernel parameter for given interface or "all"
getRPFilterValue ()
{
  command="$SSYSCTL net.ipv4.conf.$1.rp_filter 2>/dev/null" #redirect the standard error to /dev/null 
  
  output=$(/bin/sh -c "$command")
  ret=$?

  if [ $ret -ne 0 ]; then
    val=-999 #assumed value if not present, to denote the absence
  else
    command="$SECHO $output | $SAWK '{print \$3}'"
    val=$(/bin/sh -c "$command")
    ret=$?
    if [ $ret -ne 0 ]; then
      val=-999 #assumed value if not present, to denote the absence
    fi
  fi

  return $ret;
}   

#method to retrieve the network interface list
getNetInterfaceList()
{
  stage=$1
  if [ $stage = "-pre" ]; then
    # the interface list is passed in the form "eth0:130.35.64.0:PUB,eth1:139.185.44.0:PVT", parse the first private interconnect
    interfaceTuple=$2
    privNicArray=`echo $interfaceTuple | awk '{gsub(",","\n", $0); print}' | grep ":PVT" |  sed 's/\([^:]*\):\([^:]*\):PVT/\1/' | sed 's/"//g'`
  else
    #in case of -post, the fourth paramter is the crs home
    CRS_HOME=$2
    privNicArray=`$CRS_HOME/bin/oifcfg getif| $SGREP cluster_interconnect | $SAWK '{print \$1}'`
  fi
  
  ret=$?
  if [ $ret -ne 0 ]; then
    result="<RESULT>EFAIL</RESULT><EXEC_ERROR>Error while retrieving information about PRIVATE Interconnect list</EXEC_ERROR><TRACE>Unable to get the private interconnect list</TRACE><NLS_MSG><FACILITY>Prvg</FACILITY><ID>1512</ID><MSG_DATA></MSG_DATA></NLS_MSG>"
    echo $result
    exit $existstatus
  fi

  numOfPrivateNics=0
  for nicname in $privNicArray
    do
      numOfPrivateNics=`expr $numOfPrivateNics + 1`
  done 
}

#method to add interface into the bad network interface list
addNICtoBadNicList()
{
  nicToAdd=$1
  if [ "X$badNicList" = "X" ]; then
    badNicList=$nicToAdd
  else
    badNicList=$badNicList$SEPARATOR$nicToAdd
  fi
}

# we need to retrieve the network interface list depending on the stage we are in
getNetInterfaceList $1 $2

if [ $numOfPrivateNics -lt 2 ]; then
  #If there is only one private interface then we need not check the param value and hence declare success
  result="<RESULT>SUCC</RESULT><COLLECTED>0</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is correctly configured on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0452</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
  existstatus=0
  echo $result
  exit $existstatus
fi

#first check if the setting for all interfaces is done for MAX value, if yes then we can report success
#now check if the consildated value for all NICS is set correctly
name="all"
getRPFilterValue $name 
allVal=$val

if [ $allVal -eq 2 ]; then
  #the consolidated value of rp_filter is set to MAX i.e. 2 for "all" the interfaces which is SUCCESS
  result="<RESULT>SUCC</RESULT><COLLECTED>$allVal</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is correctly configured on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0452</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
  existstatus=0
  echo $result
  exit $existstatus
elif [ $allVal -eq -999 ] || [ $allVal -eq 0 ] || [ $allVal -eq 1 ]; then
  if [ $allVal -eq -999 ]; then
    # the "all" for all the interfaces is not set,
    # which means each of the individual interface now must have value of either 0 or 2
    for nicname in $privNicArray
    do
      getRPFilterValue $nicname
      ret=$?
      # make sure the command ran successfully and returned the rp_filter value for this interface
      if [ $ret -ne 0 ]; then
        # this means the value is either not set or could not be read so add this interface to bad NIC list
        addNICtoBadNicList $nicname
      else
        if [ $val -ne 0 ] && [ $val -ne 2 ]; then
          # this means the interface has rp_filter value set to something incorrect,
          # lets add this interface to bad interface list
          addNICtoBadNicList $nicname
        fi
      fi
    done
  elif [ $allVal -eq 0 ]; then
    # the consolidated value set to "all" for all the interfaces is "0",
    # which means each of the individual interface now must have value of either 0 or 2
    for nicname in $privNicArray
    do
      getRPFilterValue $nicname
      # we cannot check for return value here because it is possible that the rp_filter parameter
      # is not set for an interface in cases when the value for "all" is set, 
      # consider the value of "all" for this interface in such cases
      if [ $val -eq -999 ]; then
        val=$allVal
      fi

      if [ $val -ne 0 ] && [ $val -ne 2 ]; then
        # this means the interface has rp_filter value set to something incorrect,
        # lets add this interface to bad interface list
        addNICtoBadNicList $nicname
      fi
    done
  else
    # the consolidated value set to "all" for all the interfaces is "1",
    # which means each of the individual interface now must have rp_filter set to value 2
    for nicname in $privNicArray
    do
      getRPFilterValue $nicname
      # we cannot check for return value here because it is possible that the rp_filter parameter
      # is not set for an interface in some cases when the value for "all" is set
      # consider the value of all for this interface
      if [ $val -eq -999 ]; then
        val=$allVal
      fi

      # the val must be 2 
      if [ $val -ne 2 ]; then
        # this means the interface has rp_filter value set to something incorrect,
        # lets add this interface to bad interface list
        addNICtoBadNicList $nicname
      fi
    done
  fi
else
  # the value set to "all" is neither 0,1 or 2 and hence it is also an error condition
  result="<RESULT>VFAIL</RESULT><COLLECTED>$allVal</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not correctly configured for $name interfaces on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0453</ID><MSG_DATA><DATA>$name</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
  existstatus=2
  echo $result
  exit $existstatus
fi

if [ "X$badNicList" = "X" ]; then
  #This means all the private interconnect interfaces are having rp_filter parameter set correctly
  result="<RESULT>SUCC</RESULT><COLLECTED>$expected</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is correctly configured on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0452</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
  existstatus=0
  echo $result
  exit $existstatus
else
  result="<RESULT>VFAIL</RESULT><COLLECTED>1</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not correctly configured for interfaces $badNicList on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0453</ID><MSG_DATA><DATA>$badNicList</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
  existstatus=2
  echo $result
  exit $existstatus
fi


